<?php

namespace AlphaSelect;

class Option {

	private $db_strike;
	private $db_expiry;
	private $db_start;
	
	private $time2Expiry;
	
	private $usr_priceStart;
	private $usr_priceEnd;
	
	private $db_underlyingPricesID;
	private $db_nIDs;
	private $db_priceMult;
	private $db_priceMethod;
	
	private $db_volsID;
	private $vols;
	
	private $db_type;
	private $db_yieldsID;
	
	private $usr_optionID;
	
	private $greeks;
	private $optionPrices;
	
	private $priceDates;
	private $dNormal;
	
	private $db_data;
	private $db_volData;
	
	private $_pc;
	public $db_desc;
	
	function __construct($optionID, $startDateStr, $endDateStr, $pPacket = true){
		
		$this->usr_optionID = $optionID*1;
		if($pPacket === true){
			//Populate the rest of the parameters from the DB.
			$this->setDBParams();
			
			//Set the dates...
			$this->setDates($startDateStr, $endDateStr);
			
			//Set the DBData
			$this->setPriceData();
			$this->setVolData();
			
			//Price the option 
			$this->setOptionPricesAndGreeks();
			
		}
		//If we get sent a data packet then we'll deal with that in the if statement...TO DO.
	}
	
	/**
	 * SETS the dates needed for the option pricing.
	 * We do this separately as it could change for a given option.
	 */
	 public function setDates($sDate,$eDate){
	 	$this->setPriceEnd($eDate);
	 	$this->setPriceStart($sDate);
	 }
	/**
	 * Sets the EndPrice date for the option
	 * 
	 * Note that EndPrice is not expiry just the last date we value the option. 
	 * @param $date is a DateTime and we pre-process through the dateAdjust static method.
	 */
	public function setPriceEnd($date){
	
		//This cannot be the expiry so is forced to be the day before.
		$date = self::dateAdjust($date, $this->db_start, 'a')->format('Y-m-d');
		$this->usr_priceEnd = self::dateAdjust($date, $this->db_expiry);
	}
	
	/**
	 * Sets the StartPrice date for the option
	 * 
	 * Note that StartPrice is the first date we analyse the option on. 
	 * This set up means we can specify a range of dates to price the option. 
	 * @param $date is a DateTime and we preprocess through the dateAdjust static method. 
	 */
	public function setPriceStart($date){
		//This must be before the the priceEnd and we should check it's after the option start date...
		$date = self::dateAdjust($date, $this->db_start,'a')->format('Y-m-d');
		$this->usr_priceStart = self::dateAdjust($date, $this->usr_priceEnd);
	}
	
	/**
	 * A static function to adjust dates.
	 * 
	 * Both inputs are DateTime and then we use the modify method. 
	 * @param $adjStr gives us the scope to manipulate dates differently but we 
	 * are limited to logic that one date is bigger than the other...
	 */
	private static function dateAdjust($d1Str, $d2, $compare = 'b', $adjStr='-1day'){
		//Makes sure dBefore is before and dAfter after...
		$d1 = new \DateTime($d1Str);
		$cd2 = clone $d2;
		
		switch($compare){
			case 'a':// d1 must be at least d2
				$d1 = ($d1 < $d2) ? $d2:$d1;
				break;
			
			default: //d1 must be before d2.
				$d1 = ($d1 >= $d2) ? $cd2->modify($adjStr) : $d1;
				break;
		}
		
		return $d1;
	}
	
	/**
	 * GETS the option parameters from the db for 'real' options. 
	 */
	 private function getParams($params = null){
	 	if ($params == null){
	 		//Try to get everything first
			$mySQL = "select * from rets_series_params where series_id = {$this->usr_optionID} order by comp_order";
			$params = mxrAS_db_grabRows('AS',$mySQL,true);
	 	}

	 	return $params;
	 }
	 
	/**
	 * SETS the description - trivial if $desc is set 
	 * otherwises we grab from the db. 
	 */
	private function setDescription( $desc = null){
		if ($desc==null){
			//So we grab from the db
			//This could have a tidier format in the database...means a change to the table...
			$mySQL = "select description as d from rets_series_desc where series_id = {$this->usr_optionID}";
			$myNameArr = mxrAS_db_grabRows('AS',$mySQL,true);
			$desc = $myNameArr[0]['d'];
		}
		
		//Set the value.
		$this->db_desc = $desc;
	}
	
	/**
	 * Gets data from the database that specifies the Option.
	 * SETS the various member variables to the database data. 
	 * db_grabRows is the db call and we will make these methods into a separate class (so 
	 * we call a dbObject and set the SQL.
	 */
	private function setDBParams($desc = null, $params = null){
		
		//Set the description
		$this->setDescription($desc);
		
		//Try to get everything first
		$myParams = $this->getParams($params);
		
		//Things come in as list and should be in order comp_order
		$this->db_strike = $myParams[0]['val_num'];
		$this->db_type = strtolower($myParams[0]['val_text']);
		
		//We may expand depending on range of options we can handle...
		$this->_pc = ($this->db_type=='call') ? 0:1;
		
		//Key dates...
		$this->db_expiry = new \DateTime($myParams[1]['val_text']);
		$this->db_start = new \DateTime($myParams[4]['val_text']);
		
		//Various series ids.
        $this->db_yieldsID = array('r'=>array('ID'=>$myParams[2]['constituent'], 'L'=>$myParams[2]['val_num'],
                                        'implied'=>$myParams[2]['val_text']),
								   'q'=>array('ID'=>$myParams[3]['constituent'], 'L'=>$myParams[3]['val_num'],
								    'implied'=>$myParams[3]['val_text']));
		
		//Price multiplier and number of price IDs.
		$this->db_priceMult = $myParams[4]['val_num'];
		$this->db_nIDs = $myParams[5]['val_num'];
		
		//Price method
		$this->db_priceMethod = $myParams[5]['val_text'];
		
		//Populate IDs
		$offset = 6;

        //i.e for single underlying params will be price=6,power=7,vol=8...
		for ($i = $offset; $i < ($this->db_nIDs + $offset); $i++){
			$pKey = "{$myParams[$i]['constituent']}";
		    
			$this->db_underlyingPricesID[$pKey]['weight'] =  $myParams[$i]['val_num'];
			$this->db_underlyingPricesID[$pKey]['pow'] = $myParams[$this->db_nIDs+$i]['val_num'];
			
			//This may need a little thought in case we have some complicated expression of price...
			$vKey = $myParams[2*$this->db_nIDs + $i]['constituent'];
			$this->db_volsID["{$vKey}"] = $myParams[2*$this->db_nIDs+$i]['val_num'];
		}
    }
	
	/**
	 * GETS the dateTime members so we can chain the outputs...
	 */
	 public function getPriceStart(){
	 	return $this->usr_priceStart;
	 }
	 public function getPriceEnd(){
	 	return $this->usr_priceEnd;
	 }
	
	/**
	 * SETS data from the time-series database lets us actually price the option.
	 * Hence we need to have setDBParams first otherwise there'll be no params to find. 
	 * db_grabRows is the db call and we will make these methods into a separate class (so 
	 * we call a dbObject and set the SQL.
	 */
	private function setPriceData(){
		//We assume we have setDBParams or populated the fields so we know what data to get. 
	
		//1 - We get the data from the 'data' table...
		
		//Which IDs from the rets_data?
		$idxStrArrYields = array($this->db_yieldsID['r']['ID'],$this->db_yieldsID['q']['ID'],);
		$idxStrArr = array_keys($this->db_underlyingPricesID);
		//Build out the SQL string.
		$idxStr = implode(",",$idxStrArrYields) . "," . implode(",",$idxStrArr);
		
		//String versions of the date range.
		$dSt = $this->getPriceStart()->format('Y-m-d');
		$dEd = $this->getPriceEnd()->format('Y-m-d');
		
		$mySQL = "SELECT series_id,ts,value FROM rets_series_data WHERE series_id in ({$idxStr}) AND ts>='{$dSt}' AND ts<='{$dEd}'";
		$retsData = mxrAS_db_grabRows('AS',$mySQL,true);
				
		//2 - Now we need to reorganise this data and keep track of the 'right id'
		$ret = array();
		$addIDCount = array();
		
		foreach($retsData as $row){
			$dStr = $row['ts'];
            $method =  'XX';
            $val = $row['value'];
            $w = 1;

			//What's the key label?
			switch($row['series_id']){
				case "{$this->db_yieldsID['r']['ID']}":

                    //More complicated if we have implied yields.
                    if ($this->db_yieldsID['r']['implied']=='implied') {
                        $method = 'IMPLIED';
                        $val = $val / 1;//not 1 but the spot return which is contained in 'Price'
                        $w = 365/30;
                    }
					$this->db_data[$dStr]['yr'] = $this->manipulateReturn($val * $this->db_yieldsID['r']['L'], $w, $method);
					break;
					
				case "{$this->db_yieldsID['q']['ID']}":
                    if ($this->db_yieldsID['r']['implied']=='implied') {
                        $method = 'IMPLIED';
                        $val = $val / 1;
                        $w = 365/30;
                    }
                    $this->db_data[$dStr]['yq'] = $this->manipulateReturn($val * $this->db_yieldsID['q']['L'], $w, $method);
					break;
					
				default:
					//Otherwise the key is the series_id - this relies on the underlying data have the same number of entries...
					
					//Store the new price...
					$this->db_data[$dStr][$row['series_id']] = pow( $row['value'], $this->db_underlyingPricesID["{$row['series_id']}"]['pow']);
					
					//Grab the new return 
					$weight = $this->db_underlyingPricesID["{$row['series_id']}"]['weight'];
					
					if(!isset($ret[$dStr])){
						$ret[$dStr] = 0;
						$addIDCount[$dStr] = 0;
					}
					$ret[$dStr] += $this->manipulateReturn($this->db_data[$dStr][$row['series_id']], $weight, $this->db_priceMethod);
					$addIDCount[$dStr] += 1;//Increment it. 
					
					//Ok so we manipulated a set of prices...this really only gets right answer on last call...
					if($addIDCount[$dStr] == $this->db_nIDs){
						$this->db_data[$dStr]['finalPrice'] = $this->manipulateReturn($ret[$dStr], 1, 'END' . $this->db_priceMethod);		
					}
			}//End Switch
			
			// And we'll store the dates - but they may not be unique...
			if(!isset($this->priceDates[$dStr])){
				$this->priceDates[$dStr] = new \DateTime($dStr);
			}
			
			//We'll reset the return if we've added the right 
		} //End Loop
	}//End proc.
	
	/**
	 * Returns a price field manipulated according to the db.
	 * We have several ways driven by the option method...
	 */
	 private function manipulateReturn($val, $weight = 1, $method = 'XX'){
	 	switch ($method) {
			case 'SUM':
			//Additive, so simply weight * value
			$ret = $weight * $val;
			break;
						
			case 'GEO':
			//Geometric - i.e. a*b*c*d etc...
			$ret = log(abs($weight)*$val) * ( ($weight>0) ? 1:-1);
			break;
		
			case 'ENDGEO':
			//Helps combine a sum of the above styles...
			$val = exp($val);
			//Fall through
			
			case 'ENDSUM':
			//Adjust by the price multiplier
			$ret = $this->db_priceMult * $val;
			break;

            case 'IMPLIED':
                //We need the implied carry yield on a $weight basis
                $ret = log($val)*$weight;
			    break;
			default:
				$ret = $val;
		}
		return $ret;
	 }
	
	/**
	 * GETS the risk id and weight...
	 * We need this for the risk calculations at the portfolio level
	 */
	 public function getRiskIDWeights(){
	 	return $this->db_volsID;
	 }
	/**
	 * SETS the db_volData with data from the exantecovar table...
	 * We can combine the variances to produce the correct risk for the underlying...
	 *
	 */
	 private function setVolData(){
	 	//1 Now we need to process the vols. 
		//They are stored elsewhere so we need another db call to grab those. 
		$ids = implode("," , array_keys($this->db_volsID));
		
		//String versions of the date range.
		$dSt = $this->getPriceStart()->format('Y-m-d');
		$dEd = $this->getPriceEnd()->format('Y-m-d');
		
		$mySQL = "SELECT secid1,secid2,var,applicable_date as ddate FROM ccp_exantecovar WHERE secid1 in ({$ids}) AND secid2 in ({$ids}) AND applicable_date>='{$dSt}' AND applicable_date<='{$dEd}'";
		$rows = mxrAS_db_grabRows('AS',$mySQL,true);
		
		//2 So we have a 4 col representation of a weekly cov matrix. 
		//What is the final vol at each point in time? 
		foreach($rows as $r){
			
			//The date
			$dStr = $r['ddate'];
			
			//IDs
			$i = $r['secid1'];
			$j = $r['secid2'];
			
			//Since we only store the upper echelon we need to add the term in again for the lower, i.e. 
			$lambda = ($i!=$j) ? 2:1;
			
			//What's the variance...
			if(!isset($this->db_volData[$dStr])){
				$this->db_volData[$dStr] = 0;
			}
			$this->db_volData[$dStr] += $lambda*$this->db_volsID[$i]*$this->db_volsID[$j]*$r['var'];			
		}	
		
		//3 So we have the variances...convert to sqrt with anonymous function.
		array_walk($this->db_volData, function(&$value, &$key) {
    			$value = sqrt($value);
		});
		//Sort the array by key...
		ksort($this->db_volData);
	 }
	 
	 /** GETS the vol from db_volData for a given date...
	  * We take advantage of an array filter and then pick the 
	  * array key that's first in the list after a sort omn the keys. 
	  */ 
	 private function getVolGivenDate($date){
		
		//Filter the dates 
		$filtered_array = array();
		foreach($this->db_volData as $k=>$v){
			if($k<=$date){
				$filtered_array[$k] = $v;
			}
		}
	 	//Check to see what was returned...
	 	if(count($filtered_array)>0){
	 		//We got something
	 		end($filtered_array);
			$myDate = key($filtered_array);
	 	} else {
	 		//We have to just use the best date we have...
	 		//i.e. the first of the db_volData dates
	 		reset($this->db_volData);
	 		$myDate = key($this->db_volData);
	 	}
	 	
	 	return $this->db_volData[$myDate];
	 }
	 
	/**
	 * SETS the d1,d2 and N[d1],N[d2] values needed to price the option.
	 * Note the Option class specifies these variables as arrays because 
	 * we can value the option over a series of dates.
	 * db_grabRows is the db call and we will make these methods into a separate class (so 
	 * we call a dbObject and set the SQL.
	 */
	private function getSpotDNormal($date){
	
		//Time to expiry - in years. 
		$t = $this->time2Expiry;
		
		//Useful parameters. 
		$logITM = log($this->db_data[$date]['finalPrice'] / $this->db_strike);
		$dYield = $this->db_data[$date]['yr'] - $this->db_data[$date]['yq'];
		
		// So we want the volatility - but the volatility is weekly (because the risk model is weekly)
		// so we need to get the 'latest' figure from the voldata as volData[$date] may not be specified. 
		$vol = $this->getVolGivenDate($date);
		
		//Useful array to send back data.
		$myDN = array();
			
		//Get the probabilities
		$myDN['d1'] =  ( $logITM + $t * ( $dYield + pow($vol,2)/2) ) / ($vol * sqrt($t));
	    $myDN['d2'] = $myDN['d1'] - ($vol * sqrt($t));
			
		//And the cumulative probability distribution too...
		$myDN['Nd1'] = \PHPStats\ProbabilityDistribution\Normal::getCdf($myDN['d1']);
		$myDN['Nd2'] = \PHPStats\ProbabilityDistribution\Normal::getCdf($myDN['d2']);
		
		$myDN['t'] = $t;
			
		return $myDN;
	}
	
	/**
	 * SETS the time to expiry of the option in years.
	 * This method is called a lot - once per date in our option date range. 
	 * @params $second_per_period is the number of seconds in our default time base.
	 * Hence default is 3153600 seconds in a year. 
	 */
	private function setTimeToExpiry($date, $seconds_per_period = 31536000){
		$expiry = $this->db_expiry;
		//Returns the decimal years...
		$this->time2Expiry =  ($expiry->getTimestamp() - $date->getTimestamp())/$seconds_per_period;
	}
	
	/**
	 * GETS the Spot Option Price
	 * @param $date the date (string format) to price the option on. 
	 * @param db_data contains the $date indexed historical data. 
	 * @param _pc is the put/call type. 
	 * Currently the Garman-Kohlhagen method that gives us dividend yield OR we can use as
	 * foreign rate in FX-Options.  
	 */
	private function getSpotOptionPrice($date){
		
		//Rates for discount factors. 
		$r = $this->db_data[$date]['yr'];
		$q = $this->db_data[$date]['yq'];
		
		$t = $this->time2Expiry;
		$sq = $this->db_data[$date]['finalPrice'] * exp(-$q*$t);
		$xr = $this->db_strike * exp(-$r*$t);

        //This follows from N(-d1) = 1 - N(d1) etc.
		return $this->_pc*($xr-$sq) + $sq * $this->dNormal[$date]['Nd1']  - $xr * $this->dNormal[$date]['Nd2'];
	}
	
	/**
	 * SETS the Option Prices and Greeks
	 * We do both here to avoid lots of repeated function calls. 
	 * SET the range of dates to price the Option over...
	 * SET the type of option (put/call)...
	 * FOREACH DateTime date we use SPOT methods to build up an array of prices and greeks
	 * @param db_data contains the $date indexed historical data. 
	 * @param _pc is the put/call type. 
	 * Currently the Garman-Kohlhagen method that gives us dividend yield OR we can use as
	 * foreign rate in FX-Options.  
	 */
	private function setOptionPricesAndGreeks(){
		//We assume all the params are populated already...
		//Clear the various histories..
		$this->dNormal = array();
		
		//Loop over the history.
		foreach($this->priceDates as $date){
		    $dStr = $date->format('Y-m-d');
		    
		    //Check we haven't gone past the expiry...
		    if ($date < $this->db_expiry) {
		    
		    	//Set the time to expiry
		    	$this->setTimeToExpiry($date);
		    
		    	//Set the distribution data
		    	$this->dNormal[$dStr] = $this->getSpotDNormal($dStr);
		    
		    	//Price the option
				$this->optionPrices[$dStr] =  $this->getSpotOptionPrice($dStr);
			
				//The delta and gamma
				$this->greeks[$dStr]['delta'] = $this->getSpotDelta($dStr);
				$this->greeks[$dStr]['gamma'] = $this->getSpotGamma($dStr);
			} else {
				
				//Set the defaults post expiry.
			 	$this->dNormal[$dStr] = 0;
			 	$this->optionPrices[$dStr] = 0;
			 	$this->greeks[$dStr]['delta'] = 0;
			 	$this->greeks[$dStr]['gamma'] = 0;
			}
		}
	}
	
	/*
	 * GETS the riskDelta but does set the time2expiry so watch out after...
	 */
	public function getRiskDelta($dStr){
		$date = new \DateTime($dStr);
		$this->setTimeToExpiry($date);
		return $this->getSpotDelta($dStr);
	}
	
	/**
	 * GETS the Option Delta for a particular $date 
	 */
	private function getSpotDelta($date){
		//Rates for discount factors. 
		$Q = exp(-$this->db_data[$date]['yq']*$this->time2Expiry);
		
		return $Q*($this->dNormal[$date]['Nd1'] - $this->_pc);
			
	}
	
	/**
	 * GETS the Option Gamma for a particular $date 
	 */
	private function getSpotGamma($date){
		//This is the df 
		$Q = exp(-$this->db_data[$date]['yq']*$this->time2Expiry);
		
		//And the price fluctuation
		$D = $this->db_data[$date]['finalPrice']*$this->getVolGivenDate($date)*sqrt($this->time2Expiry);
		
		//The Boltzman coefficient. 
		$Z =  exp(-0.5*pow($this->dNormal[$date]['d1'],2));
		
		//Gamma
		return $Q*$Z/($D*sqrt(2*pi()));	
	}
	
	/**
	 * GETS the class data in the correct format for the
	 * data table.
	 */
	private function getParamsTableArr(){
		
		//Some key items...
		$expiry = $this->db_expiry;
		
		$zDates = array_keys($this->db_data);
		$date = max($zDates);
		
		$pDate = $this->getPriceEnd()->format('Y-m-d');
		
		//Set the table data...
		$tableArray = array('(1) Underlying Price'=>array(number_format($this->db_data[$date]['finalPrice'],4)),
							'(2) Strike Price'=>array(number_format($this->db_strike,4)),
							'(3) Volatility %'=>array(number_format($this->getVolGivenDate($pDate)*100,2)),
							'(4) Interest Rate %'=>array(number_format($this->db_data[$date]['yr']*100,2)),
							'(5) Dividend Yield %'=>array(number_format($this->db_data[$date]['yq']*100,2)),
							'(6) Pricing Date'=>array($pDate),
							'(7) Expiry Date'=>array($expiry->format('Y-m-d')),
							'(8) Priced From'=>array($this->getPriceStart()->format('Y-m-d'))
							);
							
		//Create the html and return.
		return $tableArray;
	} 
	
	/**
	 * Makes a chart object in the correct format - just the option price...
	 *
	 */
	public function getPriceChart($cmdName){
	    
	    $rows = array();
		foreach ($this->optionPrices as $dStr=>$price){
			$rows[] = array('x'=>strtotime($dStr)*1000,'y'=>1*number_format($price,4),'z'=>$dStr);
		}
		
		//Set the chartOptions
		$myLabelArr = explode(" Price", $this->db_desc);
		$myLabel = $myLabelArr[0];
		$chartOptions = mxrAS_HCGetChartOptions('StockChart',"{$myLabel} Price History",'Option Price','<b>{series.name}</b><br>','<b>Option Price:</b> {point.y} bp on {point.z}');
		
		$chartData = mxrAS_HCSetChartData('line', $myLabel, $rows, '', mxrAS_HCGetColors(0));
		
	   //Return the chart...
	   	$jsCharts = array();
		$jsCharts[] = array('chartOptions'=>$chartOptions,'chartData'=>$chartData);
		return $jsCharts;
	}
	
	/**
	 * Creates a table for editing in the post.
	 * Used by datatables.js client side. 
	 */
	public function getParamsTable($cmdName, $suffix = ""){
		//Make the parameters table.				
		//This will be a 2 col affair
		$cols = array('Param', 'Data');
		
		//array(1) means the whole second column will be editable.	
		$tableHtml = "<table id='mxrAS_{$cmdName}_dataTable{$suffix}' class='mxrAS_dataTableEdit'>" 
					. mxrAS_tableHeaderSimpleEdits($cols, array(0)) 
					. mxrAS_tableBodyArrayEdits($this->getParamsTableArr(), array(0), array(2), array(0))
			 		. "</table>";
	
		return array('data'=>$tableHtml, 
					 'divID'=>"mxrAS_{$cmdName}_dataTableDiv{$suffix}", 
					 'dTableParams'=>mxrAS_defaultDataTableParams(), 
					 'tDOM'=>"mxrAS_{$cmdName}_dataTable{$suffix}",
					 'msg'=>"<h4>Parameters :</h4>");
	}
	
	/*
	 * GETS the prices and greeks in the right format...
	 *
	 */
	private function getPriceGreeksArr($nRows = -1){
		
		//So each row looks like this
		// date, option_price, delta, gamma, theta, vega, rho
		$tempArr = $this->greeks;
		
		//Descending by date
		krsort($tempArr);
		
		//Then only store the ones we want...so we keep track using countRows.
		$countRows = 0;
		$nRows = ($nRows<0) ? count($tempArr) : $nRows;
		
		foreach($tempArr as $dStr=>$v){
			if ($countRows < $nRows){
				$tempArr[$dStr]['price'] = $this->optionPrices[$dStr];
				foreach($tempArr[$dStr] as $k=>&$value){
					$value = number_format($value,4);
				}
			} else {
				//quit the loop;
				break;
			}
			$countRows++;
		}
		 
		return $tempArr;
	}
	
	/**
	 * Creates a table that has the option price and greeks...
	 * Used by dataTables but its HTML...
	 */
	 public function getPriceGreeksTable($cmdName, $nRows= -1, $suffix = ""){
	 	//Make the table of historical numbers...
        $cols = array('Date', strtoupper($this->db_type) . ' Price', 'Delta', 'Gamma');//, 'Theta', 'Vega', 'Rho');
	 	
	 	//This will be a descending list (latest first) but not sortable (yet). 
	 	$tableHtml = "<h4>Prices and Greeks :</h4><table id='mxrAS_{$cmdName}_grid{$suffix}' class='mxrAS_dataTableEdit'>"
	 				. mxrAS_tableHeaderSimpleEdits($cols, array(0))
	 				. mxrAS_tableBodyArrayEdits($this->getPriceGreeksArr($nRows), array(10), array(2), array(0), ['price', 'delta', 'gamma'])
	 				. "</table>";
	 	
	 	return array('data'=>$tableHtml, 
	 				 'divID'=>"mxrAS_{$cmdName}_gridDiv{$suffix}", 
	 				 'dTableParams'=>mxrAS_defaultDataTableParams(array('sScrollY'=> "270",'bScrollCollapse'=>true)), 
	 				 'tDOM'=>"mxrAS_{$cmdName}_grid{$suffix}",
	 				 'msg'=>"");
	 }
	 
}

?>
